import cv2
from Box2D import b2PolygonShape, b2CircleShape
from Environment.Environments.Phyre.objects import Ball, Basket, Platform
import numpy as np

system_colors = {
    "green": (32, 201, 162),
    "red": (235, 82, 52),
    "blue": (32, 93, 214),
    "black": (0, 0, 0),
    "gray": (200, 200, 200),
    "purple": (81, 56, 150),
}


def transform_vertices_for_render(position, frame_width, frame_height, ppm=60):
    x, y = position
    x = int(x * ppm + frame_width / 2)  # Adjust the scaling factor as needed
    y = int(-y * ppm + frame_height / 2)  # Adjust the scaling factor as needed
    return x, y


def render_wall(wall, screen):
    screen_width, screen_height = screen.shape[0], screen.shape[1]
    for fixture in wall.fixtures:
        shape = fixture.shape
        if isinstance(shape, b2PolygonShape):
            vertices = [
                transform_vertices_for_render(
                    wall.transform * v, screen_width, screen_height
                )
                for v in shape.vertices
            ]
            cv2.fillPoly(screen, np.array([vertices]), (0, 0, 255))


def render_basket(basket_body, color, screen, ppm):
    screen_width, screen_height = screen.shape[0], screen.shape[1]
    for fixture in basket_body.fixtures:
        shape = fixture.shape
        if isinstance(shape, b2PolygonShape):
            vertices = [
                transform_vertices_for_render(
                    basket_body.transform * v, screen_width, screen_height, ppm
                )
                for v in shape.vertices
            ]
            bgr_color = (color[2], color[1], color[0])
            cv2.fillPoly(screen, np.array([vertices]), bgr_color)


def render_ball(ball, screen, ppm, color):
    screen_width, screen_height = screen.shape[0], screen.shape[1]
    for fixture in ball.fixtures:
        shape = fixture.shape
        if isinstance(shape, b2CircleShape):
            position = transform_vertices_for_render(
                ball.transform * shape.pos, screen_width, screen_height, ppm
            )
            bgr_color = (color[2], color[1], color[0])
            cv2.circle(screen, position, int(shape.radius * ppm), bgr_color, -1)


def render_platform(platform, color, screen, ppm):
    screen_width, screen_height = screen.shape[0], screen.shape[1]
    for fixture in platform.fixtures:
        shape = fixture.shape
        if isinstance(shape, b2PolygonShape):
            vertices = [
                transform_vertices_for_render(
                    platform.transform * v, screen_width, screen_height, ppm
                )
                for v in shape.vertices
            ]
            bgr_color = (color[2], color[1], color[0])
            cv2.fillPoly(screen, np.array([vertices]), bgr_color)


def render_scene(world, level, screen, ppm):
    for body in world.bodies:
        body_name = body.userData
        if "wall" in body_name:
            render_wall(body, screen)
        elif body_name in level.objects:
            color = system_colors[level.objects[body_name].color]
            if isinstance(level.objects[body_name], Basket):
                render_basket(body, color, screen, ppm)
            elif isinstance(level.objects[body_name], Platform):
                render_platform(body, color, screen, ppm)
            elif isinstance(level.objects[body_name], Ball):
                render_ball(body, screen, ppm, color)
        else:
            raise Exception(
                f"Cannot render body {body.userData}, is of unrecognized type"
            )
